const { ipcRenderer } = require('electron');
const posthog = require('posthog-js');
const path = require('path');
const logger = require(path.join(__dirname, '..', 'logger'));

let bootstrapPromise = null;
let analyticsState = {
  optedIn: true,
  userId: null,
  lastConsentUpdatedAt: null,
  initialized: false,
  config: null,
  error: null,
  sessionRecording: {
    enabled: false,
    sampleRate: 0.1
  }
};

let registryState = {
  version: null,
  updatedAt: null,
  snapshot: null
};

ipcRenderer.on('analytics.registry.updated', (_event, payload) => {
  const snapshot = payload && typeof payload === 'object' ? payload : {};
  registryState = {
    version: typeof snapshot.version === 'number' ? snapshot.version : null,
    updatedAt:
      snapshot.metadata && typeof snapshot.metadata.loadedAt === 'number'
        ? snapshot.metadata.loadedAt
        : null,
    snapshot
  };
});

function resetClient() {
  if (typeof posthog.stopSessionRecording === 'function') {
    posthog.stopSessionRecording();
  }
  if (typeof posthog.reset === 'function') {
    posthog.reset();
  }
}

function initializeClient(config, distinctId, options = {}) {
  if (!config || !config.projectApiKey) {
    resetClient();
    return null;
  }

  const sessionRecordingEnabled = Boolean(options.sessionRecordingEnabled);
  let sessionRecordingSampleRate = Number(options.sessionRecordingSampleRate);
  if (!Number.isFinite(sessionRecordingSampleRate)) {
    sessionRecordingSampleRate = 0.1;
  }
  sessionRecordingSampleRate = Math.max(0, Math.min(1, sessionRecordingSampleRate));

  posthog.init(config.projectApiKey, {
    api_host: config.apiHost || 'https://app.posthog.com',
    autocapture: false,
    capture_pageview: false,
    debug: false,
    disable_session_recording: !sessionRecordingEnabled,
    disable_persistence: false,
    persistence: 'localStorage',
    disable_cookie: true,
    advanced_disable_decide: false,
    property_blacklist: [],
    session_recording: {
      maskAllInputs: true,
      samplingRate: sessionRecordingEnabled ? sessionRecordingSampleRate : 0
    }
  });

  if (distinctId) {
    posthog.identify(distinctId);
  }

  if (sessionRecordingEnabled && typeof posthog.startSessionRecording === 'function') {
    posthog.startSessionRecording();
  } else if (!sessionRecordingEnabled && typeof posthog.stopSessionRecording === 'function') {
    posthog.stopSessionRecording();
  }

  return posthog;
}

async function bootstrapAnalytics({ force = false } = {}) {
  if (bootstrapPromise && !force) {
    return bootstrapPromise;
  }

  bootstrapPromise = (async () => {
    try {
      const response = await ipcRenderer.invoke('analytics.bootstrap');
      if (!response || response.success !== true) {
        analyticsState = {
          ...analyticsState,
          error: response && response.error ? response.error : 'Unable to bootstrap analytics.',
          initialized: false
        };
        resetClient();
        return analyticsState;
      }

      const { analytics, config } = response;
      let nextState = {
        ...analyticsState,
        ...analytics,
        config: config || null,
        initialized: true,
        error: null
      };

      analyticsState = nextState;

      if (!analyticsState.optedIn) {
        try {
          return await setAnalyticsOptIn(true, {
            rotateUserId: !analyticsState.userId,
            sessionRecordingEnabled: analyticsState.sessionRecording
              ? Boolean(analyticsState.sessionRecording.enabled)
              : false
          });
        } catch (enforcementError) {
          logger.warn('Failed to enforce analytics opt-in during bootstrap:', enforcementError && enforcementError.message ? enforcementError.message : enforcementError);
          nextState = {
            ...analyticsState,
            optedIn: true,
            error: null
          };
          analyticsState = nextState;
        }
      }

      const recordingPrefs = nextState.sessionRecording || { enabled: false, sampleRate: 0.1 };
      if (nextState.optedIn && config && config.projectApiKey) {
        initializeClient(config, nextState.userId, {
          sessionRecordingEnabled: Boolean(recordingPrefs.enabled),
          sessionRecordingSampleRate: recordingPrefs.sampleRate
        });
      } else {
        resetClient();
      }

      return analyticsState;
    } catch (error) {
      analyticsState = {
        ...analyticsState,
        error: error && error.message ? error.message : 'Failed to bootstrap analytics.',
        initialized: false
      };
      resetClient();
      return analyticsState;
    }
  })();

  return bootstrapPromise;
}

async function setAnalyticsOptIn(_optedIn, { rotateUserId = false, sessionRecordingEnabled } = {}) {
  const payload = {
    optedIn: true,
    rotateUserId: Boolean(rotateUserId)
  };

  if (typeof sessionRecordingEnabled === 'boolean') {
    payload.sessionRecordingEnabled = sessionRecordingEnabled;
  }

  const response = await ipcRenderer.invoke('analytics.preferences.set', payload);
  if (!response || response.success !== true) {
    throw new Error(response && response.error ? response.error : 'Failed to update analytics preferences.');
  }

  analyticsState = {
    ...analyticsState,
    ...response.analytics,
    error: null
  };

  const recordingPrefs = analyticsState.sessionRecording || { enabled: false, sampleRate: 0.1 };

  if (analyticsState.optedIn && analyticsState.config && analyticsState.config.projectApiKey) {
    initializeClient(analyticsState.config, analyticsState.userId, {
      sessionRecordingEnabled: Boolean(recordingPrefs.enabled),
      sessionRecordingSampleRate: recordingPrefs.sampleRate
    });
  } else {
    resetClient();
  }

  return analyticsState;
}

async function setSessionRecordingEnabled(enabled, { sampleRate } = {}) {
  const payload = {
    optedIn: Boolean(analyticsState.optedIn),
    sessionRecordingEnabled: Boolean(enabled)
  };

  if (sampleRate != null) {
    payload.sessionRecordingSampleRate = sampleRate;
  }

  const response = await ipcRenderer.invoke('analytics.preferences.set', payload);
  if (!response || response.success !== true) {
    throw new Error(response && response.error ? response.error : 'Failed to update session recording preference.');
  }

  analyticsState = {
    ...analyticsState,
    ...response.analytics,
    error: null
  };

  const recordingPrefs = analyticsState.sessionRecording || { enabled: false, sampleRate: 0.1 };

  if (analyticsState.optedIn && analyticsState.config && analyticsState.config.projectApiKey) {
    initializeClient(analyticsState.config, analyticsState.userId, {
      sessionRecordingEnabled: Boolean(recordingPrefs.enabled),
      sessionRecordingSampleRate: recordingPrefs.sampleRate
    });
  } else if (!recordingPrefs.enabled) {
    resetClient();
  }

  return analyticsState;
}

async function rotateAnalyticsIdentity() {
  const response = await ipcRenderer.invoke('analytics.identity.rotate');
  if (!response || response.success !== true) {
    throw new Error(response && response.error ? response.error : 'Failed to rotate analytics identity.');
  }

  analyticsState = {
    ...analyticsState,
    ...response.analytics,
    error: null
  };

  const recordingPrefs = analyticsState.sessionRecording || { enabled: false, sampleRate: 0.1 };

  if (analyticsState.optedIn && analyticsState.config && analyticsState.config.projectApiKey) {
    initializeClient(analyticsState.config, analyticsState.userId, {
      sessionRecordingEnabled: Boolean(recordingPrefs.enabled),
      sessionRecordingSampleRate: recordingPrefs.sampleRate
    });
  }

  return analyticsState;
}

async function capture(event, properties = {}, options = {}) {
  if (!event || typeof event !== 'string') {
    return { success: false, error: 'Event name is required.' };
  }

  if (!analyticsState.optedIn) {
    return { success: false, skipped: true };
  }

  const payload = {
    event,
    distinctId: analyticsState.userId,
    properties: {
      ...properties,
      source: options.source || 'renderer'
    },
    timestamp: options.timestamp || new Date().toISOString(),
    groups: options.groups
  };

  try {
    const response = await ipcRenderer.invoke('analytics.capture', payload);
    if (!response || response.success !== true) {
      throw new Error(response && response.error ? response.error : 'Failed to enqueue analytics event.');
    }
    if (response.skipped) {
      return { success: true, skipped: true };
    }
    if (response.dropped) {
      if (process.env.NODE_ENV !== 'production') {
        logger.warn(
          '[analytics] Event dropped by registry:',
          event,
          response.reason || 'event_not_allowed'
        );
      }
      return {
        success: true,
        dropped: true,
        reason: response.reason || 'event_not_allowed',
        registryVersion: response.registryVersion || registryState.version,
        registryUpdatedAt: response.registryUpdatedAt || registryState.updatedAt,
        diagnosticsId: response.diagnosticsId || null
      };
    }
    return {
      success: true,
      diagnosticsId: response.diagnosticsId || null,
      registryVersion: response.registryVersion || registryState.version,
      registryUpdatedAt: response.registryUpdatedAt || registryState.updatedAt
    };
  } catch (error) {
    return { success: false, error: error && error.message ? error.message : 'Failed to enqueue analytics event.' };
  }
}

function getAnalyticsState() {
  return JSON.parse(JSON.stringify(analyticsState));
}

function getPosthogClient() {
  return posthog;
}

function getAnalyticsRegistrySnapshot() {
  return registryState && registryState.snapshot ? { ...registryState.snapshot } : null;
}

module.exports = {
  bootstrapAnalytics,
  setAnalyticsOptIn,
  rotateAnalyticsIdentity,
  setSessionRecordingEnabled,
  capture,
  getAnalyticsState,
  getPosthogClient,
  getAnalyticsRegistrySnapshot
};

